/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Shared Library						*
*	Source:			BlitheDictionary.cpp						*
*	Last Modified:	5/23/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a shared library which	*
*					allows applications to have dynamic text in	*
*					different languages.						*
***********************************************************************/
#include "BlitheDictionary.h"
#include "BlitheLib.h"
#include <cstdlib>
#include <PopUpMenu.h>
#include <Path.h>
#include <FindDirectory.h>
#include <Directory.h>
#include <MenuItem.h>
#include <Application.h>
#include <NodeInfo.h>
#include <cstring>
#include <Resources.h>
#include <Bitmap.h>

// Used for consistancy
status_t blithe_init(void);

/**********************************************************************
*	Method:		BlitheDictionary (constructor)					*
*	Arguments:	int32	dictionary							*
*							the default language ID				*
*				bool		checkForAppDictionary					*
*							if true, looks for an application	*
*							dictionary for the language ID		*
*	Returns:		<none>										*
*	Description:	Sets up the Blithe dictionary object.				*
**********************************************************************/
BlitheDictionary::BlitheDictionary(int32 dictionary,bool checkForAppDictionary)
{
// Initialize functions and members
	status_t err = blithe_init();
	if (err != B_OK) return;
	appLanguage = (dictionary == BL_DEFAULT_LANGUAGE) ? get_default_language() : dictionary;
	temp = NULL;
	if (!checkForAppDictionary) return;

// Check for an app dictionary
	if (!has_application_dictionary(appLanguage)) throw BlitheException('NOAP');
}

/**********************************************************************
*	Method:		~BlitheDictionary								*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Deallocates any memory that needs it.				*
**********************************************************************/
BlitheDictionary::~BlitheDictionary(void)
{
// Free memory
	ClearTemp();
}

/**********************************************************************
*	Method:		AppLanguageID									*
*	Arguments:	<none>										*
*	Returns:		int32	the application language ID				*
*	Description:	Gets the language ID for this dictionary.			*
**********************************************************************/
inline int32 BlitheDictionary::AppLanguageID(void)
{
	return appLanguage;
}

/**********************************************************************
*	Method:		AppLanguageName								*
*	Arguments:	char*	inBuffer								*
*							optional buffer for the language name	*
*				int32	inBuffLength							*
*							number of bytes in inBuffer			*
*	Returns:		int32	String containing the language name		*
*						associated with the default language ID.	*
*	Description:	Gets the language name associated with appLanguage.	*
*				If inBuffer is null, a buffer is allocated for it.	*
*				Do not free this buffer. BlitheDictionary will do	*
*				it for you.									*
**********************************************************************/
inline char* BlitheDictionary::AppLanguageName(char *inBuffer,int32 inBuffLength)
{
	ClearTemp();
	if (inBuffer && inBuffLength) return language_name(appLanguage,inBuffer,inBuffLength);
	temp = language_name(appLanguage,NULL,0);
	return temp;
}

/**********************************************************************
*	Method:		Entry										*
*	Arguments:	int32	id									*
*							the requested entry ID				*
*				int32	dictionary							*
*							the requested language ID			*
*				char*	inBuffer								*
*							optional buffer for the string		*
*				int32	inBuffLength							*
*							number of bytes in inBuffer			*
*	Returns:		char*	String containing the text associated with*
*						entry ID id and language ID dictionary.	*
*	Description:	Gets the string associated with id and dictionary.	*
*				If inBuffer is null, a buffer is allocated for it.	*
*				Do not free this buffer. BlitheDictionary will do	*
*				it for you.									*
**********************************************************************/
char* BlitheDictionary::Entry(int32 id,int32 dictionary,char *inBuffer,int32 inBuffLength)
{
// Get the proper language ID
	switch (dictionary)
	{
	case BL_APP_LANGUAGE:
		dictionary = appLanguage;
		break;
	case BL_DEFAULT_LANGUAGE:
		dictionary = get_default_language();
		break;
	}

// Get the entry
	if (inBuffer && inBuffLength) return get_entry(id,dictionary,inBuffer,inBuffLength);
	temp = get_entry(id,dictionary,NULL,0);
	return temp;
}

/**********************************************************************
*	Method:		LanguageList									*
*	Arguments:	char**	outList								*
*							the buffer for the array of strings	*
*	Returns:		int32	Number of strings in the array			*
*	Description:	Gets the name of each language and stores it in an	*
*				array.										*
**********************************************************************/
inline int32 BlitheDictionary::LanguageList(char **outList)
{
	if (temp) free(temp);
	return get_language_list(outList);
}

/**********************************************************************
*	Method:		LanguageName									*
*	Arguments:	int32	id									*
*							the requested language ID			*
*				char*	inBuffer								*
*							optional buffer for the language name	*
*				int32	inBuffLength							*
*							number of bytes in inBuffer			*
*	Returns:		char*	String containing the language name		*
*	Description:	Gets the name of the language associated with id.	*
**********************************************************************/
inline char* BlitheDictionary::LanguageName(int32 id,char *inBuffer,int32 inBuffLength)
{
	if (id == BL_DEFAULT_LANGUAGE) id = get_default_language();
	if (inBuffer && inBuffLength) return language_name(id,inBuffer,inBuffLength);
	temp = language_name(id,NULL,0);
	return temp;
}

/**********************************************************************
*	Method:		LanguageID									*
*	Arguments:	char*	name									*
*							the requested language name			*
*	Returns:		int32	Language ID							*
*	Description:	Gets the lanauge ID associated with name.			*
**********************************************************************/
inline int32 BlitheDictionary::LanguageID(char *name)
{
	return language_id(name);
}

/**********************************************************************
*	Method:		ClearTemp										*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Frees the last memory allocation made by this class	*
**********************************************************************/
void BlitheDictionary::ClearTemp(void)
{
	if (temp)
	{
		free(temp);
		temp = NULL;
	}
}

/**********************************************************************
*	Method:		LanguagePopupMenu								*
*	Arguments:	<none>										*
*	Returns:		BPopupMenu*	Popup menu with the name of each		*
*							language in the common dictionaries.	*
*	Description:	Creates the BPopupMenu with the name of each 		*
*				available language. It is set to send a message	*
*				(BLITHE_LANGUAGE_CHANGED) to your application		*
*				object when a menu item is selected. This message	*
*				contains an int32 called langid with the new		*
*				language ID.									*
**********************************************************************/
BPopUpMenu *BlitheDictionary::LanguagePopupMenu(void)
{
	BPopUpMenu *thePopup = new BPopUpMenu("");

// Get the common dictionary directory ($B_COMMON_ETC_DIRECTORY$/Blithe)
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return thePopup;
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	if (theDirectory.InitCheck() != B_OK)
	{
		BMenuItem *theItem;
		BMessage *theMessage = new BMessage(BLITHE_LANGUAGE_CHANGED);
		theMessage->AddInt32("langid",0);
		theItem = new BMenuItem("",theMessage);
		theItem->SetTarget(be_app);
		thePopup->AddItem(theItem);
		return thePopup;
	}
	theDirectory.Rewind();
	BEntry candidate;

// Get the current language id
	int32 currentID = get_default_language();

// For each dictionary found, add a menu item with the name of the language and
// a message returning the language id
	BMenuItem *theItem = NULL;
	bool found=false;
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) == 0)
		{
			BFile theFile(&candidate,B_READ_ONLY);
			BResources theResource(&theFile);
			size_t len;
			char *name = NULL;
			int32 *id = NULL;
			name = (char *)(theResource.FindResource('CSTR',(int32)0,&len));
			id = (int32 *)(theResource.FindResource('LONG',1,&len));
			if (!name || !id) break;

		// Create the message
			BMessage *theMessage = new BMessage(BLITHE_LANGUAGE_CHANGED);
			theMessage->AddInt32("langid",*id);
			theItem = new BMenuItem(name,theMessage);
			theItem->SetTarget(be_app);

		// If the default language id equals the language id of the current file
			if (currentID == *id)
			{
				theItem->SetMarked(true);
				found = true;
			}
			free(name);
			free(id);
			thePopup->AddItem(theItem);
		}
	}
	if (!found) theItem->SetMarked(true);

	return thePopup;
}

/**********************************************************************
*	Method:		SetAppLanguage								*
*	Arguments:	int32	inLangID								*
*							the new language ID				*
*	Returns:		<none>										*
*	Description:	Sets the language ID of the BlitheDictionary to	*
*				inLangID.										*
**********************************************************************/
void BlitheDictionary::SetAppLanguage(int32 inLangID)
{
	appLanguage = inLangID;
}

/**********************************************************************
*	Method:		BlitheBitmap									*
*	Arguments:	<none>										*
*	Returns:		BBitmap*	the Blithe bitmap						*
*	Description:	Gets the Blithe bitmap							*
**********************************************************************/
BBitmap* BlitheDictionary::BlitheBitmap(void)
{
	void* theBits;
	size_t size = get_raw_bits(&theBits);
	BBitmap *theBitmap = new BBitmap(BRect(0,0,63,63),B_COLOR_8_BIT,false,true);
	theBitmap->SetBits(theBits,size,0,B_COLOR_8_BIT);
	free(theBits);
	return theBitmap;
}